/*
 * Copyright (c) 2025 Torsten Muetze
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cassert>
#include <iostream>
#include <vector>
#include "sperm.hpp"

Signed_Permutation::Signed_Permutation(int n) : n_(n)
{
  perm_.resize(n_+1);
  perm_inv_.resize(n_+1);
  signs_.resize(n_+1, +1);
  for (int j=1; j<=n_; ++j) {
    perm_[j] = j;
    perm_inv_[j] = j;
  }
  
  o_.resize(n_+1, Direction::left);
  s_.resize(n_+1, 0);
  for (int j=1; j<=n_; ++j) {
    s_[j] = j;
  }
}

bool Signed_Permutation::next()
{
  int j = s_[n_];  // the value in the permutation that performs the next transposition
  if ((j == 1) && (o_[j] == Direction::right)) {
    // the value 1 only moves once, then changes direction (left to right)
    // and then never moves again; this gives the Gray code 1, -1 for n=1
    return false;
  }
  int p = perm_inv_[j];  // position of j
  int s = signs_[p];  // sign of j
  if (((o_[j] == Direction::left) && (signs_[p] == +1)) ||
      ((o_[j] == Direction::right) && (signs_[p] == -1))) {
    if (p == 1) {
      signs_[1] = -signs_[1];  // special case of sign change of the first entry
    } else {
      // transpose with entry directly left of j
      int t = perm_[p-1];
      perm_[p-1] = perm_[p];
      perm_[p] = t;
      perm_inv_[j]--;
      perm_inv_[t]++; 
      t = signs_[p-1];
      signs_[p-1] = signs_[p];
      signs_[p] = t;
    }
  } else {
    assert(((o_[j] == Direction::left) && (signs_[p] == -1)) ||
           ((o_[j] == Direction::right) && (signs_[p] == +1)));
    // transpose with entry directly right of j
    int t = perm_[p+1];
    perm_[p+1] = perm_[p];
    perm_[p] = t;
    perm_inv_[j]++;
    perm_inv_[t]--;
    t = signs_[p+1];
    signs_[p+1] = signs_[p];
    signs_[p] = t;
  }

  s_[n_] = n_;
  if (j == 1) {  // j=1 changes direction only once
    o_[j] = Direction::right;
  } else if (o_[j] == Direction::left && (((s == +1) && (signs_[p-2]*perm_[p-2] > j)) ||
                                          ((s == -1) && ((p+1 == n_) || (-signs_[p+2]*perm_[p+2] > j))))) {
    o_[j] = Direction::right;
    s_[j] = s_[j-1];
    s_[j-1] = j-1;
  } else if ((o_[j] == Direction::right) && (((s == +1) && ((p+1 == n_) || (-signs_[p+2]*perm_[p+2] > j))) ||
                                             ((s == -1) && (signs_[p-2]*perm_[p-2] > j)))) {
    o_[j] = Direction::left;
    s_[j] = s_[j-1];
    s_[j-1] = j-1;
  }
  return true;
}

void Signed_Permutation::print()
{
  for (int i=1; i<=n_; ++i) {
    std::cout << perm_[i] << " " << (signs_[i] == +1 ? 1 : 2);
    if (i<n_) {
       std::cout << " | ";
    }
  }
  std::cout << std::endl;
}
