/*
 * Copyright (c) 2021 Jean Cardinal, Arturo Merino and Torsten Muetze
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cassert>
#include <getopt.h>
#include <iostream>
#include <utility>
#include <vector>
#include "sperm.hpp"

// display help
void help() {
  std::cout << "./sperm [options]    compute signed permutations of length n in Gray code order" << std::endl;
  std::cout << "-h                   display this help" << std::endl;
  std::cout << "-n{1,2,...}          size of the ground set" << std::endl;
  std::cout << "-l{-1,0,1,2,...}     number of permutations to generate; -1 for all" << std::endl;
  std::cout << "-q                   quiet output" << std::endl;
  std::cout << "-c                   output number of signed permutations" << std::endl;
  std::cout << "examples:  ./sperm -n3 -c" << std::endl;
  std::cout << "           ./sperm -n5 -l100" << std::endl;
  std::cout << "           ./sperm -n8 -q -c" << std::endl;
}

int main(int argc, char* argv[]) {
  int n;
  bool n_set = false;
  long long steps = -1;  // compute all signed permutations by default
  bool quiet = false;  // print output by default
  int c;
  bool output_counts = false; // omit counts by default
  int quiet_dot = 10000000;  // print one dot every 10^7 signed permutations in quiet output mode

  while ((c = getopt (argc, argv, "hn:l:qc")) != -1) {
    switch (c) {
      case 'h':
        help();
        return 0;
      case 'n':
        {
        n = atoi(optarg);
        if (n <= 0) {
          std::cerr << "option -n must be followed by an integer from {1,2,...}" << std::endl;
          return 1;
        }
        n_set = true;
        break;
        }
      case 'l':
        steps = atoi(optarg);
        if (steps < -1) {
          std::cerr << "option -l must be followed by an integer from {-1,0,1,2,...}" << std::endl;
          return 1;
        }
        break;
      case 'q':
        quiet = true;
        break;
      case 'c':
        output_counts = true;
        break;
    }
  }
  if (!n_set) {
    std::cerr << "option -n is mandatory" << std::endl;
    help();
    return 1;
  }

  int num_perms = 0;
  Signed_Permutation perm(n);

  if (steps == 0) {
    std::cout << "output limit reached" << std::endl;
    return 0;
  }

  bool next;
  do {
    num_perms++;
    if (!quiet) {
      perm.print();
    } else if (num_perms % quiet_dot == 0) {
      std::cout << "." << std::flush;
    }

    next = perm.next();
    if (next && (steps >= 0) && (num_perms >= steps)) {
      std::cout << "output limit reached" << std::endl;
      break;
    }
  } while (next);
  if (output_counts)
  {
    if (quiet && num_perms >= quiet_dot)
      std::cout << std::endl;
    std::cout << "number of signed permutations: " << num_perms << std::endl;
  }

  return 0;
}
